#if NUNIT

#region References

using System;
using System.Collections;
using System.Data;
using System.Text;
using NUnit.Framework;
using gov.va.med.vbecs.UnitTests;

using ARTIFICIAL = gov.va.med.vbecs.Common.DatabaseConstants.ArtificialColumnNames;
using PATIENT = gov.va.med.vbecs.Common.VbecsTables.Patient;
using PATIENT_CHANGE = gov.va.med.vbecs.Common.VbecsTables.PatientChange;
using TABLES = gov.va.med.vbecs.Common.VbecsTables;
using SProc = gov.va.med.vbecs.Common.VbecsStoredProcs;

using BOL = gov.va.med.vbecs.BOL;
using COMMON = gov.va.med.vbecs.Common;
using DAL = gov.va.med.vbecs.DAL;
using VAL = gov.va.med.vbecs.VAL;
using gov.va.med.vbecs.VAL.UnitTests;

#endregion

namespace gov.va.med.vbecs.DAL.HL7AL.UnitTests
{
	[TestFixture]
	public class HL7AL_PatientChange : BaseTest
	{
		private DataTable _dtToPatient;
		
		private const string SELECT_STATEMENT = "SELECT TOP 1 * FROM Patient " +
			"WHERE PatientGuid NOT IN " +
			"(SELECT FromPatientGuid FROM PatientChange WHERE FromPatientGuid IS NOT NULL " +
			"UNION " +
			"SELECT ToPatientGuid FROM PatientChange WHERE ToPatientGuid IS NOT NULL)";

		/// <summary>
		/// SetUp
		/// </summary>
		[SetUp]
		protected void SetUp()
		{
			this.BaseSetUp();
			//
			_dtToPatient = PatientChange.GetPatientChangeDataTable();
			_dtToPatient.Rows.Add ( _dtToPatient.NewRow() );
			_dtToPatient.Rows[0][Common.VbecsTables.PatientChange.PatientChangeGuid] = System.Guid.NewGuid();
			_dtToPatient.Rows[0][Common.VbecsTables.PatientChange.PatientChangeStatusCode] = "P";
			_dtToPatient.Rows[0][Common.VbecsTables.PatientChange.LastUpdateUser] = "Unit Test";
			Common.Utility.AppendLastUpdateInformation(_dtToPatient, Common.UpdateFunction.UnitTests);
		}

		/// <summary>
		/// TearDown
		/// </summary>
		[TearDown]
		protected void TearDown()
		{
		}

		/// <summary>
		/// InsertPatientChange - PASS
		/// </summary>
		[Test]
		public void InsertPatientChange_Pass()
		{
			DataRow drPatient = UnitTestUtil.RunSQLGetDataRow( SELECT_STATEMENT );
			string patientId = null;
			if( !drPatient.IsNull(Common.VbecsTables.Patient.VistaPatientId) )
			{
				patientId = drPatient[Common.VbecsTables.Patient.VistaPatientId].ToString();
			}
			else
				Assert.Fail("No Patient available for unit testing");
			//
			Guid patientGuid = (Guid)drPatient[Common.VbecsTables.Patient.PatientGuid];
			//
			string fromPatientFirstName = drPatient[Common.VbecsTables.Patient.PatientFirstName].ToString();
			string toPatientFirstName = string.Empty;
			//
			if ( fromPatientFirstName != null && fromPatientFirstName != string.Empty && fromPatientFirstName.Length > 0 )
			{
				toPatientFirstName = string.Concat( fromPatientFirstName[fromPatientFirstName.Length - 1], fromPatientFirstName.Substring( 1, fromPatientFirstName.Length - 1) );
			}
			//
			if (fromPatientFirstName == toPatientFirstName)
			{	
				toPatientFirstName = "AAABBBCCCDDD";
			}
			//
			_dtToPatient.Rows[0][Common.VbecsTables.PatientChange.FromPatientGuid] = patientGuid;
			_dtToPatient.Rows[0][Common.VbecsTables.PatientChange.ToPatientGuid] = patientGuid;
			_dtToPatient.Rows[0][Common.VbecsTables.PatientChange.FromPatientFirstName] = fromPatientFirstName;
			_dtToPatient.Rows[0][Common.VbecsTables.PatientChange.ToPatientFirstName] = toPatientFirstName;
			//
			HL7AL.PatientChange.InsertPatientChange(_dtToPatient.Rows[0]);
			//
			string query = "SELECT ToPatientFirstName FROM PatientChange WHERE ToPatientGuid = '" + patientGuid + "'";
			//
			DataRow drUpdate = UnitTestUtil.RunSQLGetDataRow(query);
			//
			Assert.IsTrue(drUpdate[Common.VbecsTables.PatientChange.ToPatientFirstName].ToString() == toPatientFirstName);
		}

		/// <summary>
		/// InsertPatientChange - FAIL
		/// </summary>
		[Test]
		[ExpectedException(typeof(ArgumentNullException))]
		public void InsertPatientChange_Fail()
		{
			PatientChange.InsertPatientChange(null);
		}

		/// <summary>
		/// GetPatientChangeDataTable - PASS
		/// </summary>
		[Test]
		public void GetPatientChangeDataTable_Pass()
		{
			Assert.IsNotNull(PatientChange.GetPatientChangeDataTable());
		}

		/// <summary>
		/// GetPatientChangeDataTable - FAIL
		/// </summary>
		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void GetPatientChangeDataTable_Fail()
		{}

		/// <summary>
		/// GetPatientUpdateDataTable - PASS
		/// </summary>
		[Test]
		public void GetPatientUpdateDataTable_Pass()
		{
			Assert.IsNotNull(PatientChange.GetPatientUpdateDataTable());
		}

		/// <summary>
		/// GetPatientUpdateDataTable - FAIL
		/// </summary>
		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void GetPatientUpdateDataTable_Fail()
		{}

		/// <summary>
		/// UpdatePatientInformation - PASS
		/// </summary>
		[Test]
		public void UpdatePatientInformation_Pass()
		{
			ArrayList storedProcedureList = new ArrayList();
			ArrayList updatedTableList = new ArrayList();
			//
			DataRow drPatient = UnitTestUtil.RunSQLGetDataRow( "SELECT TOP 1 * FROM Patient" );
			Assert.IsFalse( drPatient.IsNull("PatientGuid") );
			Assert.IsFalse( drPatient.IsNull("VistaPatientId") );
			//
			Guid patientGuid = (Guid)drPatient["PatientGuid"];
			string vistaPatientId = drPatient["VistaPatientId"].ToString();
			//
			DataRow drFromPatient = HL7AL.CprsOmgMessage.GetHl7Patient( null, vistaPatientId ).Rows[0];
			//
			DataTable dtPatientChange = HL7AL.PatientChange.GetPatientChangeDataTable();
			DataRow drPatientChange = dtPatientChange.NewRow();
			//
			drPatientChange[PATIENT_CHANGE.PatientChangeStatusCode] = "P";
			drPatientChange[PATIENT_CHANGE.PatientChangeGuid] = System.Guid.NewGuid();
			drPatientChange[PATIENT_CHANGE.FromPatientGuid] = patientGuid;
			drPatientChange[PATIENT_CHANGE.ToPatientGuid] = patientGuid;
			//
			DataTable dtPatientUpdate = HL7AL.PatientChange.GetPatientUpdateDataTable();
			DataRow drPatientUpdate = dtPatientUpdate.NewRow();
			//
			foreach( System.Data.DataColumn dc in dtPatientUpdate.Columns)
			{
				if( drFromPatient.Table.Columns.Contains(dc.ColumnName) )
				{
					drPatientUpdate[dc.ColumnName] = drFromPatient[dc.ColumnName];
				}
			}
			//
			drPatientChange[PATIENT_CHANGE.ToPatientLastName] = "CHANGED";
			drPatientUpdate[PATIENT.PatientLastName] = "CHANGED";
			drPatientChange[PATIENT_CHANGE.FromPatientLastName] = drFromPatient[PATIENT.PatientLastName];
			//
			string lastUpdateUser = "UNIT TEST";
			//
			drPatientUpdate[PATIENT.LastUpdateUser] = lastUpdateUser;
			drPatientUpdate[PATIENT.LastUpdateFunctionId] = Common.UpdateFunction.UnitTests;
			dtPatientUpdate.Rows.Add( drPatientUpdate );
			//
			drPatientChange[PATIENT_CHANGE.LastUpdateUser] = lastUpdateUser;
			drPatientChange[PATIENT_CHANGE.LastUpdateFunctionId] = Common.UpdateFunction.UnitTests;
			dtPatientChange.Rows.Add( drPatientChange );
			//
			storedProcedureList.Add( SProc.HL7InsertPatientChange.StoredProcName );	
			storedProcedureList.Add( SProc.HL7UpdatePatient.StoredProcName );	
			//
			updatedTableList.Add( dtPatientChange.Copy() );
			updatedTableList.Add( dtPatientUpdate.Copy() );
			//
			Assert.IsTrue( HL7AL.PatientChange.UpdatePatientInformation(storedProcedureList, updatedTableList) );
		}

		/// <summary>
		/// UpdatePatientInformation - FAIL
		/// </summary>
		[Test]
		[ExpectedException(typeof(ArgumentException))]
		public void UpdatePatientInformation_Fail_1()
		{
			ArrayList storedProcedureList = new ArrayList();
			ArrayList updatedTableList = new ArrayList();
			//
			// sprocCount < 1 
			HL7AL.PatientChange.UpdatePatientInformation(storedProcedureList, updatedTableList);
		}

		/// <summary>
		///  UpdatePatientInformation - FAIL
		/// </summary>
		[Test]
		[ExpectedException(typeof(ArgumentException))]
		public void UpdatePatientInformation_Fail_2()
		{
			ArrayList storedProcedureList = new ArrayList();
			ArrayList updatedTableList = new ArrayList();
			//
			storedProcedureList.Add( new DataTable() );
			storedProcedureList.Add( new DataTable() );
			storedProcedureList.Add( new DataTable() );
			storedProcedureList.Add( new DataTable() );
			//
			// sprocCount > 3
			HL7AL.PatientChange.UpdatePatientInformation(storedProcedureList, updatedTableList);
		}

		/// <summary>
		///  UpdatePatientInformation - FAIL
		/// </summary>
		[Test]
		[ExpectedException(typeof(ArgumentException))]
		public void UpdatePatientInformation_Fail_3()
		{
			ArrayList storedProcedureList = new ArrayList();
			ArrayList updatedTableList = new ArrayList();
			//
			storedProcedureList.Add( new DataTable() );
			//
			// updatedTableList < 1
			HL7AL.PatientChange.UpdatePatientInformation(storedProcedureList, updatedTableList);
		}

		/// <summary>
		///  UpdatePatientInformation - FAIL
		/// </summary>
		[Test]
		[ExpectedException(typeof(ArgumentException))]
		public void UpdatePatientInformation_Fail_4()
		{
			ArrayList storedProcedureList = new ArrayList();
			ArrayList updatedTableList = new ArrayList();
			//
			storedProcedureList.Add( new DataTable() );
			updatedTableList.Add( new DataTable() );
			updatedTableList.Add( new DataTable() );
			updatedTableList.Add( new DataTable() );
			updatedTableList.Add( new DataTable() );
			//
			// updatedTableList > 3
			HL7AL.PatientChange.UpdatePatientInformation(storedProcedureList, updatedTableList);
		}

		/// <summary>
		/// IsVbecsPatient - PASS
		/// </summary>
		[Test]
		public void IsVbecsPatient_Pass_Patient_In_VBECS()
		{
			DataRow drDfn = UnitTestUtil.RunSQLGetDataRow( "SELECT TOP 1 VistaPatientId FROM Patient" );
			//
			long vistaPatientId = Convert.ToInt64( drDfn["VistaPatientId"] );
			//
			Assert.IsTrue( PatientChange.IsVbecsPatient(vistaPatientId) );
		}

		/// <summary>
		/// IsVbecsPatient - PASS
		/// </summary>
		[Test]
		public void IsVbecsPatient_Pass_Patient_Not_In_VBECS()
		{
			bool dfnFound = false;
			long counter = 0;
			string baseQuery = "SELECT COUNT(1) AS COUNT FROM Patient WHERE VistaPatientId = ";
			//
			while (!dfnFound)
			{
				counter += 1;
				//
				DataRow drDfn = UnitTestUtil.RunSQLGetDataRowDontThrowExceptionIfNoRowIsReturned( baseQuery + counter );
				//
				if ( Convert.ToInt64( drDfn["Count"] ) == 0 )
				{
					dfnFound = true;
				}
			}
			//
			Assert.IsFalse( PatientChange.IsVbecsPatient(counter) );
		}

		/// <summary>
		/// IsVbecsPatient - FAIL
		/// </summary>
		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void IsVbecsPatient_Fail()
		{}
	}
}
#endif
